package mcfall.raytracer.tests;

import mcfall.math.ColumnVector;
import mcfall.math.Matrix;
import mcfall.math.Point;
import mcfall.math.Vector;
import junit.framework.TestCase;

public class Ray extends TestCase {

	private mcfall.math.Ray fortyFiveDegreeFromOrigin;
	private Point origin;
	private Point oneTwoThree;
	private Vector fortyFiveDegreeAngle;
	
	public Ray() {
		origin = new Point (0.0, 0.0, 0.0);
		fortyFiveDegreeAngle = new ColumnVector (3);
		fortyFiveDegreeAngle.setValueAt(1, 1);
		fortyFiveDegreeAngle.setValueAt(2, 1);
		fortyFiveDegreeAngle.setValueAt(3, 1);
		
		fortyFiveDegreeFromOrigin = new mcfall.math.Ray (origin, fortyFiveDegreeAngle);
		oneTwoThree = new Point (1, 2, 3);
	}

	protected void setUp() throws Exception {
		super.setUp();
	}

	/*
	 * Test method for 'mcfall.raytracer.Ray.Ray(Point, Point)'
	 */
	public void testRayPointPoint() {
		mcfall.math.Ray testRay = new mcfall.math.Ray (origin, oneTwoThree);
		Point start = testRay.getStart();
		assertIsOrigin(start);
		
		Vector direction = testRay.getDirection();
		assertEquals (1.0, direction.getValueAt(1));
		assertEquals (2.0, direction.getValueAt(2));
		assertEquals (3.0, direction.getValueAt(3));
		assertEquals (Math.sqrt(14.0), direction.length());
	}

	

	/*
	 * Test method for 'mcfall.raytracer.Ray.Ray(Point, Vector)'
	 */
	public void testRayPointVector() {
		Point start = fortyFiveDegreeFromOrigin.getStart();
		assertIsOrigin(start);
		
		Vector direction = fortyFiveDegreeFromOrigin.getDirection();
		assertEquals (1.0, direction.getValueAt(1));
		assertEquals (1.0, direction.getValueAt(2));
		assertEquals (1.0, direction.getValueAt(3));
	}

	/*
	 * Test method for 'mcfall.raytracer.Ray.Ray(Ray)'
	 */
	public void testRayRay() {
		mcfall.math.Ray copy = new mcfall.math.Ray (fortyFiveDegreeFromOrigin);
		assertIsOrigin(copy.getStart());
		Vector direction = copy.getDirection();
		assertEquals (1.0, direction.getValueAt(1));
		assertEquals (1.0, direction.getValueAt(2));
		assertEquals (1.0, direction.getValueAt(3));
	}

	/*
	 * Test method for 'mcfall.raytracer.Ray.pointAt(double)'
	 */
	public void testPointAt() {		
		assertIsOrigin(fortyFiveDegreeFromOrigin.pointAt(0.0));
		Point pointAtOne = fortyFiveDegreeFromOrigin.pointAt(1.0);
		assertEquals (1.0, pointAtOne.getX());
		assertEquals (1.0, pointAtOne.getY());
		assertEquals (1.0, pointAtOne.getZ());
		
		Point pointAtOneHalf = fortyFiveDegreeFromOrigin.pointAt(0.5);
		assertEquals (0.5, pointAtOneHalf.getX());
		assertEquals (0.5, pointAtOneHalf.getY());
		assertEquals (0.5, pointAtOneHalf.getZ());
		
		Point pointAtMinusOne = fortyFiveDegreeFromOrigin.pointAt(-1.0);
		assertEquals (-1.0, pointAtMinusOne.getX());
		assertEquals (-1.0, pointAtMinusOne.getY());
		assertEquals (-1.0, pointAtMinusOne.getZ());
		
	}

	/*
	 * Test method for 'mcfall.raytracer.Ray.transform(Matrix)'
	 */
	public void testTransform() {
		//  First create a ray that starts at the origin and extends out by 1, 1, 1
		mcfall.math.Ray ray = new mcfall.math.Ray (origin, fortyFiveDegreeAngle);

		//  Now translate the ray by (tx, ty, tz)
		double tx = 1.0;
		double ty = 2.0;
		double tz = 3.0;		
		Matrix translate = Matrix.createTranslationMatrix(tx, ty, tz);
		mcfall.math.Ray transformedRay = ray.transform(translate);
		Point transformedStart = transformedRay.getStart();
		Vector transformedDirection = transformedRay.getDirection();
		
		//  Check that the new start is (tx, ty, tz) 		
		assertEquals (tx, transformedStart.getX());
		assertEquals (ty, transformedStart.getY());
		assertEquals (tz, transformedStart.getZ());
				
		//  And that the vector hasn't been changed
		assertEquals (1.0, transformedDirection.getValueAt (1));
		assertEquals (1.0, transformedDirection.getValueAt (2));
		assertEquals (1.0, transformedDirection.getValueAt (3));
		assertEquals (0.0, transformedDirection.getValueAt (4));
		
		//  Now scale the resulting ray by 3, 2, 1
		double sx = 3;
		double sy = 2;
		double sz = 1;
		Matrix scale = Matrix.createScalingMatrix (sx, sy, sz);
		transformedRay = transformedRay.transform(scale);
		transformedStart = transformedRay.getStart();
		transformedDirection = transformedRay.getDirection();		
		assertEquals (sx*tx, transformedStart.getX());
		assertEquals (sy*ty, transformedStart.getY());
		assertEquals (sz*tz, transformedStart.getZ());
		assertEquals (1.0, transformedStart.getValueAt(4));
		
		assertEquals (sx, transformedDirection.getValueAt (1));
		assertEquals (sy, transformedDirection.getValueAt (2));
		assertEquals (sz, transformedDirection.getValueAt (3));
		assertEquals (0.0, transformedDirection.getValueAt (4));		
	}

	private void assertIsOrigin(Point start) {
		assertEquals (0.0, start.getX());
		assertEquals (0.0, start.getY());
		assertEquals (0.0, start.getZ());
	}
}
